<?php

namespace Drupal\taxonomy_domain_404\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\domain\DomainNegotiatorInterface;
use Drupal\taxonomy\TermInterface;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Routing\RouteMatchInterface;

class TaxonomyDomain404Subscriber implements EventSubscriberInterface {

  protected $entityTypeManager;
  protected $domainNegotiator;
  protected $themeManager;
  protected $cache;

  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    DomainNegotiatorInterface $domainNegotiator,
    ThemeManagerInterface $themeManager,
    CacheBackendInterface $cache
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->domainNegotiator = $domainNegotiator;
    $this->themeManager = $themeManager;
    $this->cache = $cache;
  }

  public static function getSubscribedEvents() {
    return [
      KernelEvents::REQUEST => ['onKernelRequest'],
    ];
  }
  
  public function onKernelRequest(RequestEvent $event) {
    // Solo aplicamos en frontend (no en CLI ni en admin).
    if (!$event->isMainRequest()) {
      return;
    }

    // Verificar si el tema activo es el deseado.
    $active_theme = $this->themeManager->getActiveTheme()->getName();
    if ($active_theme !== 'mitema') {
      return;
    }

    $route_match = \Drupal::routeMatch();

    // Solo ejecutar en rutas de términos de taxonomía.
    if ($route_match->getRouteName() !== 'entity.taxonomy_term.canonical') {
      return;
    }

    $term = $route_match->getParameter('taxonomy_term');

    if ($term instanceof TermInterface) {
      $term_id = $term->id();

      $domain = $this->domainNegotiator->getActiveDomain();
      $domain_id = $domain ? $domain->id() : 'none';

      $cid = "term_has_content:{$term_id}:{$domain_id}";
      $cache = $this->cache->get($cid);

      if ($cache) {
        $has_content = $cache->data;
      }
      else {
        // Ejecutar entityQuery solo si no hay cache.
        $taxonomy_fields = ['field_mis_temas_1', 'field_mis_temas_2'];

        $query = $this->entityTypeManager->getStorage('node')->getQuery();
        $query->condition('status', 1);

        $or_group = $query->orConditionGroup();
        foreach ($taxonomy_fields as $field_name) {
          $or_group->condition($field_name, $term_id);
        }
        $query->condition($or_group);

        if ($domain_id !== 'none') {
          // Campo general asignado a los nodos para gestionar el dominio.
          $query->condition('field_domain_access', $domain_id);
        }

        $nids = $query->accessCheck(TRUE)->range(0, 1)->execute();

        $has_content = !empty($nids);

        // Guardar en cache y asociar a la cache tag del término.
        $this->cache->set(
          $cid,
          $has_content,
          CacheBackendInterface::CACHE_PERMANENT,
          ['taxonomy_term:' . $term_id]
        );
      }

      if (!$has_content) {
        throw new NotFoundHttpException();
      }
    }
  }
}
